<?php
//
//  Copyright (c) 2016 NetApp, Inc.
//  All rights reserved.
//

//
// This check insures that before upgrading a cluster in a SAN environment, every volume
// containing LUNs has available at least 1 MB of free space. The space is needed to
// accommodate changes in the on-disk data structures used by the new version of Data ONTAP.
//
// Note - There are no volume attributes to use to determine if the volume contains LUNs.  We must
//        therefore use the LUN table to see the associated vols and then only consider those
//	  unique vols during the verification process.
//

include 'mcc_util.php';

//
// Need at least 1MB of reserve space in a vol for upgrading LUNs (1024 x 1024).
//
$MIN_AVAILABLE_MEM = (1024*1024);

//
// Helper to log debug messages
//
function logDebugMessage($message) {
    $dbgquery = "CALL add_scripting_log_message('validation_rules_lun_reserve_check: $message','false')";
    $result = dotsql_query($dbgquery);
}

//
// Helper to warn user
//
function warnUser()
{
    echo "<Alert>";
    echo "Before upgrading a cluster in a SAN environment, every volume containing ";
    echo "LUNs must have at least 1MB of available free space. The space is needed to ";
    echo "accommodate changes in the on-disk data structures used by the new version ";
    echo "of Data ONTAP. At least one volume with less than 1MB of available space ";
    echo "has been detected with associated LUNs.";
    echo "</Alert><Action>";
    echo "Run the 'volume show -available <1M' command to identify volumes that have ";
    echo "less than 1MB of available space.  Next, run the 'lun show -field volume' ";
    echo "command to determine which of those volumes have associated LUNs. ";
    echo "Finally, increase the amount of free space on those volumes before attempting to upgrade.";
    echo "</Action><Advice>";
    echo "Ensure the volumes containing LUNs have at least 1MB available space before attempting to upgrade.";
    echo "</Advice>";
    echo "<RC>error</RC>";
}

//
// All the validations only apply for major upgrades.
//
$options = getoptreq('', array("major:", "downgrade::"));
$majorupgrade = $options["major"];
if ($majorupgrade == "false") {
    // exit here because this is a minor upgrade.
    exit;
}

//
// Find all volumes with less than 1MB available space and a state valid for LUN support.
// For example, volumes associated a destination metrocluster verver that is stopped will
// show up as neither online or offline but rather '-'.  Looking for this vol state was
// chosen to make the algorithm very quick and not needing to transverse any other tables
// like metrocluster,vserver etc.
//
$query_vols = "SELECT available FROM volume WHERE (available < $MIN_AVAILABLE_MEM) and (state != '-')";
$result_vols = dotsql_query($query_vols);
if (dotsql_num_rows($result_vols) == 0) {
    logDebugMessage("All volumes have enough available memory for SAN upgrade\n");
    exit;
}

//
// Find first volume that has an associated LUN, if found exit.
//
while ($row = dotsql_fetch_object($result_vols)) {
    $query_lun = "SELECT lun WHERE volume = $row->volume";
    $result_query_lun = dotsql_query($query_lun);
    if (dotsql_num_rows($result_query_lun) == 0) {
	warnUser();
	exit;
    }
}

exit;
?>
