<?php
//
//  Copyright (c) 2014-2015 NetApp, Inc.
//  All rights reserved.
//

// VERSION="1.0"
// NDU Compatibility for snapmirror
//
// For DP relationships, the script warns the user about the following scenarios:
// - Possible protection outage in case of intra-cluster relationships.
// - In case of inter-cluster relationships with remote source, the script warns
//   the user about protection outage if the remote cluster is not upgraded
//   first.
// - In case of inter-cluster relationships with both remote source and remote
//   destination, the script warns the user about possible cyclic dependencies
//   and protection outage that could occur because of it.
//
// For LS relationships, the script warns the user that some load sharing
// mirrors might not get updated while the cluster upgrade is in progress.

include 'mcc_util.php';

$intraClusterFound = false;
$remoteSrcFound = false;
$remoteDstNames;
$lsRelsFound = false;

function checkDPRelations() {
    global $intraClusterFound;
    global $remoteSrcFound;
    global $remoteDstNames;

    // quick check to see if there are any relationships at all
    $result = dotsql_query("SELECT * FROM localSrcSnapMirror WHERE type = DP LIMIT 1");
    if (dotsql_num_rows($result) == 0) {
	$result = dotsql_query("SELECT * FROM localDstSnapMirror WHERE type = DP LIMIT 1");
	if (dotsql_num_rows($result) == 0) {
	    // there are no SM relationships, so there is no more validation to
	    // be done; exit
	    return;
	}
    }

    // look for intra-cluster SM relationships. To do this find at least on
    // entry in localSrcSnapMirror with dstVsUuid of a local Vserver.

    $localVservers = dotsql_query("SELECT uuid FROM vserver WHERE type = data");
    while ($vserver = dotsql_fetch_object($localVservers)) {
	$result = dotsql_query(
		"SELECT *" .
		" FROM localSrcSnapMirror" .
		" WHERE dstVsUuid = " . $vserver->{'uuid'} . " AND type = DP" .
		" LIMIT 1");

	if (dotsql_num_rows($result) > 0) {
	    // we've found a intra-cluster DP relationship
	    $intraClusterFound = true;
	    break;
	}
    }
    unset($localVservers);

    // Now, we need to check if there is at least on inter-cluster SM relationship
    // with this cluster as the destination. For this, find the first relationship
    // in localDstSnapMirror with a remote vserver UUID.
    //
    // Also, we need to make a list of remote cluster that need to be upgraded first.
    // To do this, for each peered cluster, we need to check if there is at least
    // one entry in the localSrcSnapMirror for one of its constituent vservers.

    $peeredClusters = dotsql_query("SELECT cluster-uuid, cluster FROM clusterPeer_idTable");
    while ($cluster = dotsql_fetch_object($peeredClusters)) {
	$peeredVservers = dotsql_query(
	    "SELECT peer-vserver-uuid".
	    " FROM vserverPeer_rdb_tbl".
	    " WHERE peer-clusterID = " . $cluster->{'cluster-uuid'} .
	    " AND applications = snapmirror");

	// a remote vserver can be peered to more than one local server, so it can
	// appear several times in the query above. $processedVservers will be used
	// to avoid processing a remote vserver more than once.
	$processedVservers;

	$remoteDstFound = false;
	// for each remote vserver, find at least one relationship with remote
	// source and one relationship with remote destination.
	while (($remoteDstFound == false || $remoteSrcFound == false) &&
		$vserver = dotsql_fetch_object($peeredVservers)) {

	    if (array_key_exists($vserver->{'peer-vserver-uuid'},
				 $processedVservers)) {
		// we have checked for this vserver already, so move on to the next
		// vserver
		continue;
	    } else {
		$processedVservers[$vserver->{'peer-vserver-uuid'}] = 1;
	    }

	    if ($remoteDstFound == false) {
		// check for inter-cluster relationships with remote destination
		$result = dotsql_query(
		    "SELECT *" .
		    " FROM localSrcSnapMirror" .
		    " WHERE dstVsUuid = " . $vserver->{'peer-vserver-uuid'} .
			" AND type = DP" .
		    " LIMIT 1");

		if (dotsql_num_rows($result) > 0) {
		    // add the cluster to the list of clusters that need to be
		    // upgraded first.
		    if (!empty($remoteDstNames)) {
			$remoteDstNames = $remoteDstNames . ", ";
		    }
		    $remoteDstNames = $remoteDstNames . $cluster->{cluster};
		    $remoteDstFound = true;
		}
	    }

	    if ($remoteSrcFound == false) {
		// check for inter-cluster relationships with remote source
		$result = dotsql_query(
		    "SELECT *" .
		    " FROM localDstMirrorBySrcVserver" .
		    " WHERE srcVsUuid = " . $vserver->{'peer-vserver-uuid'} .
			" AND type = DP" .
		    " LIMIT 1");

		if (dotsql_num_rows($result) > 0) {
		    // we have found a record!
		    $remoteSrcFound = true;
		}
	    }
	}
    }
}

function checkLSRelations() {
    global $lsRelsFound;

    $result = dotsql_query("SELECT * FROM snapmirrorTable WHERE type = LS LIMIT 1");
    $lsRelsFound = false;
    if (dotsql_num_rows($result) > 0) {
	// we've found a LS relationship
	$lsRelsFound = true;
    }
}

// All the validations only apply for major upgrades
$options = getoptreq('', array("major:", "downgrade::"));
$majorupgrade = $options["major"];
if($majorupgrade == "false") {
    // exit here because this is a minor upgrade
    exit;
}

# check for DP relationships
checkDPRelations();

# check for LS relationships
checkLSRelations();

if ($intraClusterFound == false &&
    empty($remoteDstNames) &&
    $lsRelsFound == false)
{
    // no warnings; it's all good
    exit;
}

// frame the action/advice message:
$alert = "";
$advice = "";
if ($intraClusterFound == true) {
    // warn the user about protection outage for intra-cluster relationships
    $advice = "For intra-cluster DP SnapMirror relationships, you may lose" .
	      " protection for the duration of the upgrade.";

    $alert = "DP";
}

if (!empty($remoteDstNames)) {
    // advice the user to upgrade the destination clusters first.
    if (!empty($advice)) {
	$advice = $advice . " ";
    }
    $advice = $advice . "For inter-cluster DP SnapMirror relationships, upgrade" .
	      " the following destination clusters first before upgrading the" .
	      " current cluster: " .  $remoteDstNames . "." .
	      " If the current cluster is upgraded first, you will lose" .
	      " protection until the destination clusters are upgraded, too.";

    $alert = "DP";
}

if (!empty($remoteDstNames) && $remoteSrcFound == true) {
    // warn the user about possible cyclic relationships
    if (!empty($advice)) {
	$advice = $advice . " ";
    }
    $advice = $advice . "If the cluster hosts both source and destination" .
		" volumes for different sets of DP SnapMirror relationships," .
		" you will lose data protection for all the relationships for" .
		" which the current cluster is the source until the destination" .
		" cluster is upgraded, too.";

    $alert = "DP";
}

if (true == $lsRelsFound) {
    if (!empty($advice)) {
	$advice = $advice . " ";
    }
    $advice = $advice . "For LS SnapMirror relationships, mirror copies might" .
		" not get updated while the cluster is being upgraded." .
		" Mirroring will resume once all the nodes in the cluster have" .
		" been upgraded. It is recommended that all load-sharing mirror" .
		" copies be updated manually before and after the upgrade" .
		" process using the \"snapmirror update-ls-set\" command.";

    if (!empty($alert)) {
        $alert = $alert . " and LS";
    } else {
        $alert = "LS";
    }
}

// dump out the messages
$alert = $alert . " SnapMirror relationships detected.";
echo "<Alert>" . $alert . "</Alert>";
echo "<Action>" . $advice . "</Action>";
echo "<Advice>" . $advice . "</Advice>";
echo "<RC>warning</RC>";
exit;
?>
